/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: x509InputCertificate.c,v 1.13 2001/01/25 22:11:39 jeffc Exp $
____________________________________________________________________________*/
/*
 *  Author: michael_elkins@nai.com
 *	Last Edit: January 11, 2000
 */

#include "x509CMS.h"
#include "pgpX509Priv.h"
#include "pgpKeyPriv.h"
#include "pkcsreq_asn.h"
#include "pkcsreq_oid.h"

/* convert hex value into integer */
#define hexval(c) (((c)>='0'&&(c)<='9')?(c)-'0':10+(c)-'A')

static PGPKeyDBObjRef
x509FindPrivateKey (
	PGPContextRef	context,
	PGPKeyDBRef		db,
	PKICONTEXT		*asncontext,
	PKISignerInfos	*si)
{
	int				i, asnerr = 0;
	PGPByte			*hashval = 0;
	PGPSize			hashlen = 0;
	PGPKeyDBObjRef	key = 0;

	/* we should only have one signer */
	pgpAssert (si->n <= 1);
	if (si->n==0 || !si->elt[0])
		return NULL;
	/* find the at-pki-transaction-id authenticated attribute */
	pgpAssert (si->elt[0]->authenticatedAttributes != 0);
	if (!si->elt[0]->authenticatedAttributes)
		return NULL;
	pgpAssert (si->elt[0]->authenticatedAttributes->n > 0);
	for (i = 0; i < si->elt[0]->authenticatedAttributes->n; i++)
	{
		if (memcmp (si->elt[0]->authenticatedAttributes->elt[i]->type.val,
				PKIat_pki_transactionid_OID,
				PKIat_pki_transactionid_OID_LEN) == 0)
		{
			/* depending on the CA, this will either be an INTEGER or
			   PrintableString */

			if (*si->elt[0]->authenticatedAttributes->elt[i]->values.elt[0]->val == PKIID_PrintableString)
			{
				PKIPrintableString *ps = 0;

				PKIUnpackPrintableString (asncontext,
					&ps,
					si->elt[0]->authenticatedAttributes->elt[i]->values.elt[0]->val,
					si->elt[0]->authenticatedAttributes->elt[i]->values.elt[0]->len,
					&asnerr);

				if (asnerr == 0)
				{
					PGPSize j;

					/* convert the hex string into binary */
					hashlen = ps->len / 2;
					hashval = PGPNewData (PGPPeekContextMemoryMgr (context),
						ps->len / 2, 0);
					for (j = 0; j < hashlen; j++)
					{
						hashval[j] = (hexval(ps->val[2 * j]) << 4) | hexval(ps->val[2 * j + 1]);
					}
					PKIFreePrintableString (asncontext, ps);
				}
			}
			else
			{
				PKIINTEGER *asn;

				PKIUnpackINTEGER (asncontext, &asn,
					si->elt[0]->authenticatedAttributes->elt[i]->values.elt[0]->val,
					si->elt[0]->authenticatedAttributes->elt[i]->values.elt[0]->len,
					&asnerr);

				if (asnerr == 0)
				{
					hashval = asn->val;
					hashlen = asn->len;
					PGPFreeData (asn);
				}
			}

			break;
		}
	}

	if (hashval)
	{
		PGPFilterRef	filter = 0;
		PGPKeySetRef	keyset = 0;
		PGPKeyIterRef	iter = 0;
		PGPError		err;

		err = PGPNewKeyDBObjDataFilter (context, kPGPKeyProperty_X509MD5Hash,
			hashval, hashlen, kPGPMatchCriterion_Equal, &filter);
		if (IsPGPError (err))
			goto error;
		err = PGPFilterKeyDB (db, filter, &keyset);
		if (IsPGPError (err))
			goto error;
		err = PGPNewKeyIterFromKeySet (keyset, &iter);
		if (IsPGPError (err))
			goto error;
		err = PGPKeyIterNextKeyDBObj (iter, kPGPKeyDBObjType_Key, &key);

error:

		if (iter)
			PGPFreeKeyIter (iter);
		if (filter)
			PGPFreeFilter (filter);
		if (keyset)
			PGPFreeKeySet (keyset);

		PGPFreeData (hashval);
	}

	return key;
}

static void
x509VerifyPKCS7Signatures (
	PGPContextRef	context,
	PGPKeyDBRef		keydb,			/* [IN] where to find keys */
	PKICONTEXT		*asnContext,	/* [IN] */
	PKISignedData	*sigData,		/* [IN] */
	PGPBoolean		*sigPresent,	/* [OUT] */
	PGPBoolean		*sigChecked,	/* [OUT] */
	PGPBoolean		*sigVerified,	/* [OUT] */
	PGPKeyDBObjRef	*signKey		/* [OUT] */
)
{
    X509CMSCallbackData	pgpData;
    int			p7err;
    int			asnError = 0;
    PGPError		err;
    PGPFilterRef	filter = NULL;
    PGPKeySetRef	targetKeySet = NULL;
    PGPKeyIterRef	keyIter = NULL;
    PGPKeyDBObjRef	keyRef = NULL;
    PGPByte		*isnData = NULL;
    PGPSize		isnDataLen;
    PKICertificate	*cert = NULL;
    PGPMemoryMgrRef	mem = PGPPeekContextMemoryMgr (context);

	if (!sigData || sigData->signerInfos.n < 1 || !sigData->signerInfos.elt[0])
		return; /* nothing to check */

	*sigPresent = 1;

	if (!keydb)
		return; /* Can't check without keys to check with */

	/* find certificate for this user */
	/* use the Issuer And Serial Number as a filter */
	isnDataLen = PKISizeofIssuerAndSerialNumber (asnContext,
		&sigData->signerInfos.elt[0]->issuerAndSerialNumber,
		1);
	isnData = PGPNewData (mem, isnDataLen, 0);
	PKIPackIssuerAndSerialNumber (asnContext,
		isnData,
		isnDataLen,
		&sigData->signerInfos.elt[0]->issuerAndSerialNumber,
		&asnError);
	if (asnError)
		goto error_exit;

	err = PGPNewKeyDBObjDataFilter( context, kPGPSigProperty_X509IASN, 
				isnData, isnDataLen, kPGPMatchCriterion_Equal, &filter );
	if (IsPGPError (err))
		goto error_exit;
	err = PGPFilterKeySet (pgpKeyDBPeekRootSet(keydb), filter, &targetKeySet);
	if (IsPGPError (err))
		goto error_exit;
	err = PGPNewKeyIterFromKeySet (targetKeySet, &keyIter);
	if (IsPGPError (err))
		goto error_exit;
	err = PGPKeyIterNextKeyDBObj (keyIter, kPGPKeyDBObjType_Key, &keyRef);
	if (IsPGPError (err))
		goto error_exit;

	*signKey = keyRef;

	err = x509ExtractCertificate (context,
		keyRef,
		asnContext,
		&cert);
	if (IsPGPError (err))
		goto error_exit;

	*sigChecked = 1;

	/* set up callback data */
	memset (&pgpData, 0, sizeof (pgpData));
	pgpData.context = context;
	pgpData.key = keyRef;

	/* verify signature */
	p7err = sm_VerifyMessage (NULL,	/* dont return the inner data */
		0,
		sigData,
		cert,
		pkcs7HashCallback,
		(void *) &pgpData,
		pkcs7VerifyCallback,
		(void *) &pgpData,
		asnContext);
	if (p7err == PKCS7_OK)
		*sigVerified = 1;

error_exit:

	if (keyIter)
		PGPFreeKeyIter (keyIter);
	if (targetKeySet)
		PGPFreeKeySet (targetKeySet);
	if (filter)
		PGPFreeFilter (filter);
	if (cert)
		PKIFreeCertificate (asnContext, cert);
	if (isnData)
		PGPFreeData (isnData);
}

static void
x509ExtractCertificatesFromSignedData (
	PKICONTEXT					*asnContext,
	PKISignedData				*sigData,
	PKIExtendedCertificatesAndCertificates	*certSet)
{
	int i;

	if (sigData->certificates)
	{
		for (i = 0; i < sigData->certificates->n; i++)
		{
			PKIAddOfElement(asnContext,sigData->certificates->elt[i],certSet);
			sigData->certificates->elt[i] = NULL;
		}
		/* don't free this data when free'ing sigData */
		if (sigData->certificates->n)
			PGPFreeData( sigData->certificates->elt );
		PGPFreeData (sigData->certificates);
		sigData->certificates = NULL;
	}
}

static void
x509ExtractCRLsFromSignedData (
	PKICONTEXT						*asnContext,
	PKISignedData					*sig,
	PKICertificateRevocationLists	*crls)
{
	int i;

	if (sig->crls)
	{
		for (i = 0; i < sig->crls->n; i++)
		{
			PKIAddOfElement (asnContext, sig->crls->elt[i], crls);
			sig->crls->elt[i] = NULL;
		}
		if (sig->crls->n)
			PGPFreeData (sig->crls->elt);
		PGPFreeData (sig->crls);
		sig->crls = NULL;
	}
}

static PGPError
x509ParseAsCertificate(
	PKICONTEXT *asnContext,
	/*const*/ PGPByte *input,
	PGPSize inputSize,
	PKIExtendedCertificatesAndCertificates *certSet)
{
	/* first check to see if this is a vanilla X.509 certificate */
	int asnError = 0;
	PKICertificate *cert;
	PKIExtendedCertificateOrCertificate *extended;

	PKIUnpackCertificate (asnContext, &cert, input, inputSize, &asnError);
	if(asnError)
		return kPGPError_LazyProgrammer;
	extended = PKINewExtendedCertificateOrCertificate (asnContext);
	extended->CHOICE_field_type = PKIID_Certificate;
	extended->data = (void *) cert; /* NOTE: don't free `cert' */
	PKIAddOfElement (asnContext, extended, certSet);
	return kPGPError_NoErr;
}

static PGPError
x509ParseAsCRL(
	PKICONTEXT *asnContext,
	/*const*/ PGPByte *input,
	PGPSize inputSize,
	PKICertificateRevocationLists *crls)
{
	/* check for x.509 crl */
	int asnError=0;
	PKICertificateList *crl;

	PKIUnpackCertificateList(asnContext,&crl,input,inputSize,&asnError);
	if(asnError)
		return kPGPError_LazyProgrammer;
	PKIAddOfElement (asnContext, crl, crls); /* don't free crl */
	return kPGPError_NoErr;
}

PGPError
X509InputCertificate (
	PGPContextRef	context,
	PGPByte		*input,		/* input buffer */
	PGPSize		inputSize,
	PGPKeyDBRef	keydb,		/* where to find keys */
	PGPInputFormat	format,		/* Import format (CA) */
	PGPOptionListRef    passphrase,/* Passphrase for decrypting */
	PGPKeyDBObjRef	*decryptKey,	/* Output decryption key */
	PGPKeyDBObjRef	*signKey,	/* Output signing key */
	PGPBoolean	*sigPresent,	/* output flag for signed message */
	PGPBoolean	*sigChecked,	/* output that we have sig key */
	PGPBoolean	*sigVerified,	/* output flag for sig ok */
	PGPAttributeValue	**extraData,
	PGPUInt32	*extraDataCount,
	PGPByte		**certOut,
	PGPSize		*certOutSize,
	PGPByte		**crlOut,
	PGPSize		*crlOutSize
)
{
	/* The calling function expects to be returned SET OF Certificate, but
	   the PKCS-7 definition ExtendedCertificatesAndCertificates produces the
	   same encoding when no ExtendedCertificate's are present, so just use
	   that defintion to reduce code size */
	PKIExtendedCertificatesAndCertificates	*certSet = NULL;

	PKICONTEXT					asnContext;
	PKIContentInfo				*contentInfo = NULL;
	PKISignedData				*sigData = NULL;
	PKIEnvelopedData			*envData = NULL;
	PKICertificateRevocationLists		*crls = NULL;
	PGPError					err;
	PGPMemoryMgrRef				mem = PGPPeekContextMemoryMgr (context);
	PGPByte					*contentData = NULL;
	PGPSize					contentDataLen;
	int						messageType;
	int						asnError = 0;

	(void) passphrase;	/* Unused argument */
	(void) format;	/* Unused argument */

	/* clear outputs */
	*sigPresent = 0;
	*sigChecked = 0;
	*sigVerified = 0;
	*extraData = NULL;
	*extraDataCount = 0;
	*decryptKey = NULL;
	*signKey = NULL;
	*crlOut = NULL;
	*crlOutSize = 0;

	memset (&asnContext, 0, sizeof (PKICONTEXT));
	asnContext.memMgr = &X509CMSMemoryMgr;
	asnContext.memMgr->customValue = (void *) mem;

	certSet = PKINewExtendedCertificatesAndCertificates (&asnContext);
	crls = PKINewCertificateRevocationLists(&asnContext);

	/* first check to see if input is a plain x.509 certificate.  if not, check
	   to see if input is a plain CRL.  failing that, attempt to decode as
	   a PKCS-7 message */
	if (x509ParseAsCertificate(&asnContext,input,inputSize,certSet)!=kPGPError_NoErr &&
		x509ParseAsCRL(&asnContext,input,inputSize,crls)!=kPGPError_NoErr)
	{
		/* either this is not a certificate/crl or it has a bad encoding, so
		   try to decode it as a PKCS-7 message */

		contentInfo = sm_DecodeMessage (input, inputSize, &asnContext);
		if (!contentInfo)
		{
			/* failed to decode message */
			err = kPGPError_InvalidPKCS7Encoding;
			goto error_exit;
		}

		messageType = sm_MessageType (contentInfo, &asnContext);
		if (!contentInfo->content)
		{
			PKIFreeContentInfo (&asnContext, contentInfo);
			err = kPGPError_InvalidPKCS7Encoding;
			goto error_exit;
		}

		/* save the internal content */
		contentData = contentInfo->content->val;
		contentDataLen = contentInfo->content->len;

		contentInfo->content->val = NULL;
		contentInfo->content->len = 0;
		PGPFreeData (contentInfo->content);
		contentInfo->content = NULL;

		PKIFreeContentInfo (&asnContext, contentInfo);

		/* remove the security enhancements */
		while (messageType == PKCS7_CONTENT_SIGNED_DATA ||
			messageType == PKCS7_CONTENT_ENVELOPED_DATA)
		{
			if (messageType == PKCS7_CONTENT_SIGNED_DATA)
			{
				sigData = sm_DecodeSignedData (contentData,
					contentDataLen,
					&asnContext);
				if (!sigData)
				{
					err = kPGPError_InvalidPKCS7Encoding; /* parse error */
					goto error_exit;
				}

				/* find our key based on the transaction-id in the
				   authenticated attributes */
				if (!*decryptKey)
				{
					*decryptKey = x509FindPrivateKey (context, keydb,
						&asnContext, &sigData->signerInfos);
				}

				/* check for signature.  since the returned certificate is
				   itself signed, we don't exit upon failure during this
				   step */
				x509VerifyPKCS7Signatures (
					context,
					keydb,
					&asnContext,
					sigData,
					sigPresent,
					sigChecked,
					sigVerified,
					signKey);

				/* if there are certificates present, return them */
				x509ExtractCertificatesFromSignedData (&asnContext, sigData,
													   certSet);

				/* if there are CRLs present, return them */
				x509ExtractCRLsFromSignedData (&asnContext, sigData, crls);

				PGPFreeData (contentData); /* free existing data */

				/* get the content data */
				sm_VerifyMessage (&contentData,
					&contentDataLen,
					sigData,
					NULL,
					NULL,
					NULL,
					NULL,
					NULL,
					&asnContext);

				/* peek at the internal content type */
				messageType = sm_ContentType (&sigData->contentInfo.contentType, &asnContext);

				PKIFreeSignedData (&asnContext, sigData);
				sigData = NULL;
			}
			else
			{
				/* EnvelopedData */

				X509CMSCallbackData	pgpData;

				pgpAssert (messageType == PKCS7_CONTENT_ENVELOPED_DATA);

				envData = sm_DecodeEnvelopedData (contentData, contentDataLen,
					&asnContext);
				if (!envData)
				{
					err = kPGPError_InvalidPKCS7Encoding;
					goto error_exit;
				}

				PGPFreeData (contentData);
				contentData = 0;

				memset (&pgpData, 0, sizeof (pgpData));
				pgpData.passphrase = passphrase;
				pgpData.context = context;
				pgpData.key = *decryptKey; /* my private key */

				if (sm_DecryptMessage (&contentData,
						&contentDataLen,
						envData,
						NULL, /* use first recipient */
						pkcs7DecryptCallback,
						(void *) &pgpData,
						&asnContext) != PKCS7_OK)
				{
					err = kPGPError_PKCS7DecryptFailure;
					goto error_exit;
				}

				/* peek at the internal content type */
				messageType = sm_ContentType (&envData->encryptedContentInfo.contentType,
					&asnContext);

				PKIFreeEnvelopedData (&asnContext, envData);
				envData = NULL;
			}

			/* Netscape CMS 4.1 also encrypts the message, so do a quick
			   check to see if this look like an encrypted message.  we do
			   this by looking to see if this looks like the beginning of
			   an ASN.1 SEQUENCE OF */
			if (messageType == PKCS7_CONTENT_DATA &&
				contentDataLen && *contentData == 0x30)
			{
				/* probably ASN.1 */
				contentInfo = sm_DecodeMessage (contentData, contentDataLen,
					&asnContext);
				if (contentInfo)
				{
					/* yes, it really is */

					/* set the correct message type */
					messageType = sm_MessageType (contentInfo, &asnContext);

					/* replace the current content with the interior content */
					PGPFreeData (contentData);
					if (contentInfo->content)
					{
						contentData = contentInfo->content->val;
						contentDataLen = contentInfo->content->len;

						/* so PKIFreeContentInfo doesn't free the memory */
						PGPFreeData (contentInfo->content);
						contentInfo->content = NULL;
					}
					else
					{
						contentData = NULL;
						contentDataLen = 0;
					}

					PKIFreeContentInfo (&asnContext, contentInfo);
				}
			}
		}

		if (messageType != PKCS7_CONTENT_DATA)
		{
			/* unknown PKCS7 content type */
			err = kPGPError_InvalidPKCS7Encoding;
			goto error_exit;
		}

		/* if we have something from Verisign, the certificates are located
		   in the body of the signed message */
		if (contentData)
		{
			PKICertRep *certRep;

			/* Try to parse with VeriSign structure (extra reginfo) */
			asnError = 0;
			PKIUnpackCertRep (&asnContext,
				&certRep,
				contentData,
				contentDataLen,
				&asnError);

			if (certRep) {
				/* Verisign parse succeeded, extract contentInfo portion */
				contentInfo = certRep->certs;
				certRep->certs = NULL;
				PKIFreeCertRep (&asnContext, certRep);
			} else {
				/* Try to parse as a plain CMS message */
				asnError = 0;
				contentInfo = sm_DecodeMessage (contentData, contentDataLen,
												&asnContext);
			}

			if (contentInfo)
			{
				messageType = sm_MessageType (contentInfo, &asnContext);
				if (messageType == PKCS7_CONTENT_SIGNED_DATA &&
					contentInfo->content)
				{
					PKIUnpackSignedData (&asnContext, &sigData,
										 contentInfo->content->val,
										 contentInfo->content->len,
										 &asnError);
					if (!asnError)
					{
						x509ExtractCertificatesFromSignedData (&asnContext,
															   sigData,
															   certSet);
						x509ExtractCRLsFromSignedData (&asnContext, sigData,
													   crls);
						PKIFreeSignedData (&asnContext, sigData);
						sigData = NULL;
					}
				}
				PKIFreeContentInfo (&asnContext, contentInfo);
			}
		}
	}

	asnError = 0; /* reset error code */

	if (certSet->n)
	{
		*certOutSize = PKISizeofExtendedCertificatesAndCertificates (
			&asnContext,
			certSet,
			1);
		*certOut = PGPNewData (mem, *certOutSize, 0);
		asnError = 0;
		PKIPackExtendedCertificatesAndCertificates (&asnContext,
			*certOut,
			*certOutSize,
			certSet,
			&asnError);
		if (asnError)
		{
			PGPFreeData (*certOut);
			*certOut = NULL;
			*certOutSize = 0;
			err = kPGPError_ASNPackFailure;
			goto error_exit;
		}
	}

	if (crls->n)
	{
		/* if we found CRLs, pack them up now */
		*crlOutSize = PKISizeofCertificateRevocationLists (&asnContext, crls, 1);
		*crlOut = PGPNewData (mem, *crlOutSize, 0);
		PKIPackCertificateRevocationLists (&asnContext, *crlOut, *crlOutSize, crls, &asnError);
		if (asnError)
		{
			PGPFreeData (*crlOut);
			*crlOut = NULL;
			*crlOutSize = 0;

			PGPFreeData (*certOut);
			*certOut = NULL;
			*certOutSize = 0;

			err = kPGPError_ASNPackFailure;
			goto error_exit;
		}
	}

	err = kPGPError_NoErr;

error_exit:

	/* certSet and crls should always have been allocated */
	PKIFreeExtendedCertificatesAndCertificates (&asnContext, certSet);
	PKIFreeCertificateRevocationLists(&asnContext, crls);

	if (envData)
		PKIFreeEnvelopedData (&asnContext, envData);
	if (sigData)
		PKIFreeSignedData (&asnContext, sigData);
	if (contentData)
		PGPFreeData (contentData);
	if (passphrase)
		PGPFreeOptionList (passphrase);

	return err;
}

/* vim:ts=4:sw=4:
 */
